<?php
// Include database connection
include 'conn.php';

// Initialize variables
$error_message = '';
$user_details = null;

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // --- 1. Retrieve form data and Sanitize/Validate (Crucial for security) ---
    // NOTE: The original HTML form fields were hidden, making the login process
    // ineffective. I assume the actual login form is on a different page that
    // posts to this script, or there's a typo in the original HTML.
    // For this PHP code to work, the POST variables must be present.
    // In the HTML below, I've made the fields visible for a proper login form.
    
    // Simple sanitization
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';

    // Basic validation
    if (empty($phone) || empty($password)) {
        $error_message = "Please enter both phone number and password.";
    } else {
        // --- 2. SQL query to check credentials (Using prepared statements is good) ---
        // ⚠️ SECURITY WARNING: The original code uses a plain text password comparison.
        // For production, passwords MUST be hashed using password_hash() and checked
        // using password_verify(). Since the original code didn't use hashing,
        // I'll stick to the original logic but recommend fixing it immediately.
        $sql = "SELECT phone, name, location, branch FROM user WHERE phone = ? AND password = ?";
        
        $stmt = $conn->prepare($sql);
        // "ss" indicates two string parameters
        $stmt->bind_param("ss", $phone, $password); 
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows > 0) {
            // Credentials are correct, fetch user details
            $user_details = $result->fetch_assoc();

            // --- 3. COOKIE HEADER CODE (Requested by user) ---
            // Set cookies for user details.
            // These cookies will last for 30 days (time() + 60*60*24*30).
            // NOTE: Cookies are sent as HTTP headers *before* any HTML output.
            setcookie('user_phone', $user_details['phone'], time() + (86400 * 30), "/"); // 86400 = 1 day
            setcookie('user_name', $user_details['name'], time() + (86400 * 30), "/");
            // You can set other details as needed, but be mindful of sensitive info.
            
            // --- 4. JavaScript/Redirect ---
            // Set local storage values in JavaScript (as in the original code)
            // Using JSON encoding for safer string handling in JS context
            $js_script = "
                <script>
                    localStorage.setItem('phone', '" . addslashes($user_details['phone']) . "');
                    localStorage.setItem('name', '" . addslashes($user_details['name']) . "');
                    localStorage.setItem('location', '" . addslashes($user_details['location']) . "');
                    localStorage.setItem('branch', '" . addslashes($user_details['branch']) . "');
                    
                    window.location.href = 'home.php'; // Redirect to the next page
                </script>";
            
            // Echo the script and exit to ensure immediate redirection/storage update
            echo $js_script;
            exit; // Stop further execution after redirect
            
        } else {
            $error_message = "Invalid phone number or password. Please try again.";
        }

        $stmt->close();
    }
}

// Close connection at the end of PHP logic
if (isset($conn)) {
    $conn->close();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vetline Services - Delivery Portal</title>
    <style>
        /* Modern App Style CSS */
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap');

        :root {
            --primary-color: #04837f; /* Teal/Green for branding */
            --secondary-color: #2e7d32; /* Darker Green */
            --accent-color: #ffd700; /* Yellow accent */
            --bg-color: #f4f6f8;
            --card-bg: #ffffff;
            --text-color: #333333;
            --error-color: #e53935;
        }

        body {
            font-family: 'Inter', sans-serif;
            background-color: var(--bg-color);
            color: var(--text-color);
            margin: 0;
            padding: 0;
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100vh;
            /* Simple, subtle app-like background gradient (optional) */
            background: linear-gradient(135deg, var(--bg-color) 0%, #e0e4e7 100%);
        }
        
        .container {
            background-color: var(--card-bg);
            border-radius: 16px; /* Rounded corners for modern look */
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1); /* Deeper shadow */
            padding: 30px;
            text-align: center;
            width: 90%;
            max-width: 380px;
            animation: slideUp 0.8s ease-out;
        }
        
        @keyframes slideUp {
            from { opacity: 0; transform: translateY(50px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .logo-section {
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid #eee;
        }

        .app-title {
            color: var(--primary-color);
            font-size: 24px;
            font-weight: 700;
            margin-top: 10px;
        }

        h1 {
            color: var(--secondary-color);
            font-size: 28px;
            margin-bottom: 15px;
        }

        form {
            display: flex;
            flex-direction: column;
            align-items: center;
            width: 100%; /* Important for internal elements */
        }

        label {
            align-self: flex-start; /* Align labels to the left */
            font-weight: 600;
            margin-top: 15px;
            font-size: 14px;
            color: var(--text-color);
        }
        
        .input-group {
            width: 100%;
            text-align: left;
        }

        input[type="text"], input[type="password"] {
            width: 100%;
            box-sizing: border-box; /* Include padding/border in the element's total width */
            padding: 12px 15px;
            margin: 8px 0 15px;
            border: 1px solid #ddd;
            border-radius: 8px; /* App-like rounding */
            background-color: #fafafa;
            color: var(--text-color);
            transition: border-color 0.3s, box-shadow 0.3s;
        }
        
        input[type="text"]:focus, input[type="password"]:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(4, 131, 127, 0.2);
            outline: none;
        }

        button {
            background-color: var(--primary-color); /* Main CTA color */
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 50px; /* Pill shape */
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            transition: background-color 0.3s ease, transform 0.2s;
            margin-top: 25px;
            width: 100%;
        }

        button:hover {
            background-color: var(--secondary-color);
            transform: translateY(-2px);
        }
        
        .error {
            color: var(--error-color);
            margin-top: 15px;
            font-size: 14px;
            background-color: #ffebee;
            padding: 10px;
            border-radius: 5px;
            border-left: 3px solid var(--error-color);
        }
        
        .welcome-message {
            font-size: 18px;
            color: var(--secondary-color);
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="logo-section">
            <img src="img/logo/merlogo.png" alt="Logo" style="width:35%; border-radius:8px; box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1);">
            <div class="app-title">Delivery Portal</div>
        </div>

        <?php if ($user_details): ?>
            <h1 class="welcome-message">Welcome Back, <?php echo htmlspecialchars($user_details['name']); ?>! 👋</h1>
            <form action="home.php" method="get"> <p>Your delivery hub: <strong><?php echo htmlspecialchars($user_details['branch']); ?></strong></p>
                <p>Location: <strong><?php echo htmlspecialchars($user_details['location']); ?></strong></p>
                <button type="submit">GO TO DASHBOARD</button>
            </form>
        <?php else: ?>
            <h1>Sign In</h1>
            <form method="post" action=""> <div class="input-group">
                    <label for="phone">Phone Number</label>
                    <input type="text" id="phone" name="phone" required placeholder="e.g., 0712345678" value="<?php echo htmlspecialchars($phone ?? ''); ?>">
                </div>
                
                <div class="input-group">
                    <label for="password">Password</label>
                    <input type="password" id="password" name="password" required placeholder="Enter your password">
                </div>
        
                <button type="submit">LOGIN</button>
                
                <?php if ($error_message): ?>
                    <div class="error"><?php echo htmlspecialchars($error_message); ?></div>
                <?php endif; ?>
            </form>
        <?php endif; ?>
    </div>
</body>
</html>
